<?php

/**
 * Elementor Widget
 * @package Kiante
 * @since 1.0.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class kiante_post_grid_slides extends Widget_Base {


	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'posts-grid-slides';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Posts Slider Small', 'kiante-extra' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'fas fa-th-large'; 
	}





	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'kiante_widgets' ];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	 
	
	protected function register_controls() {
		$this->layout_options();		
		$this->post_query_options();	
		$this->meta_options();	
		$this->design_options();
	}
	
	/**
    * Layout Options
    */
    private function layout_options() {
	
	
		$this->start_controls_section(
            'layout_option',
            [
                'label' => __( 'Layout Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

		
		
		$this->add_responsive_control(
			'grid_img_height',
			[
				'label' =>esc_html__( 'Set Post Image Height', 'kiante-extra' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 130,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 130,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 130,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 130,
				],
				
				'selectors' => [
					'{{WRAPPER}} .post-grid-wrapper-small-inner .smallgrid-thumbnail-one-wrap img' => 'height: {{SIZE}}{{UNIT}}!important;',
				],
			]
		);		
		
		
		$this->end_controls_section();
	
	}
	
	/**
    * Post Query Options
    */
    private function post_query_options() {
	
	
		$this->start_controls_section(
            'post_query_option',
            [
                'label' => __( 'Post Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		// Post Sort
		
        $this->add_control(
            'post_sorting',
            [
                'type'    => Controls_Manager::SELECT2,
                'label' => esc_html__('Post Sorting', 'kiante-extra'),
                'default' => 'date',
                'options' => [
					'date' => esc_html__('Recent Post', 'kiante-extra'),
                    'rand' => esc_html__('Random Post', 'kiante-extra'),
					'title'         => __( 'Title Sorting Post', 'kiante-extra' ),
                    'modified' => esc_html__('Last Modified Post', 'kiante-extra'),
                    'comment_count' => esc_html__('Most Commented Post', 'kiante-extra'),
					
                ],
            ]
        );		
		
		// Post Order
		
        $this->add_control(
            'post_ordering',
            [
                'type'    => Controls_Manager::SELECT2,
                'label' => esc_html__('Post Ordering', 'kiante-extra'),
                'default' => 'DESC',
                'options' => [
					'DESC' => esc_html__('Desecending', 'kiante-extra'),
                    'ASC' => esc_html__('Ascending', 'kiante-extra'),
                ],
            ]
        );
		
		
		// Post Categories
		
		$this->add_control(
            'post_categories',
            [
                'type'      => Controls_Manager::SELECT2,
				'label' =>esc_html__('Select Categories', 'kiante-extra'),
                'options'   => $this->posts_cat_list(),
                'label_block' => true,
                'multiple'  => true,
            ]
        );
	
		
		
		// Post Items.
		
        $this->add_control(
            'post_number',
			[
				'label'         => esc_html__( 'Number Of Posts', 'kiante-extra' ),
				'type'          => Controls_Manager::NUMBER,
				'default'       => '1',
			]
        );
		
		$this->add_control(
            'enable_offset_post',
            [
               'label' => esc_html__('Enable Skip Post', 'kiante-extra'),
               'type' => Controls_Manager::SWITCHER,
               'label_on' => esc_html__('Yes', 'kiante-extra'),
               'label_off' => esc_html__('No', 'kiante-extra'),
               'default' => 'no',
               
            ]
        );
      
        $this->add_control(
			'post_offset_count',
			  [
			   'label'         => esc_html__( 'Skip Post Count', 'kiante-extra' ),
			   'type'          => Controls_Manager::NUMBER,
			   'default'       => '1',
			   'condition' => [ 'enable_offset_post' => 'yes' ]

			  ]
        );
		
		
		$this->end_controls_section();
	
	}	
	
	/**
    * Meta Options
    */
    private function meta_options() {
	
	
		$this->start_controls_section(
            'meta_option',
            [
                'label' => __( 'Meta Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		$this->add_control(
            'display_title',
            [
                'label' => esc_html__('Display Post Title', 'ennlil-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_excerpt',
            [
                'label' => esc_html__('Display Post Excerpt', 'ennlil-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );

		$this->add_control(
         	'display_author',
         	[
				 'label' => esc_html__('Display Author', 'kiante-extra'),
				 'type' => Controls_Manager::SWITCHER,
				 'label_on' => esc_html__('Yes', 'kiante-extra'),
				 'label_off' => esc_html__('No', 'kiante-extra'),
				 'default' => 'yes',
         	]
     	);

     	$this->add_control(
            'display_cat',
            [
                'label' => esc_html__('Display Category Name', 'kiante-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'kiante-extra'),
                'label_off' => esc_html__('No', 'kiante-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_date',
            [
                'label' => esc_html__('Display Date', 'kiante-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'kiante-extra'),
                'label_off' => esc_html__('No', 'kiante-extra'),
                'default' => 'yes',
            ]
        );
		
		

	
		$this->end_controls_section();
	
	}	
	
	/**
    * Design Options
    */
    private function design_options() {
	
	
		$this->start_controls_section(
            'design_option',
            [
                'label' => __( 'Slider Typography', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'title_typography',
              'label' => esc_html__( 'Post Title Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-grid-wrapper-small-inner .post-grid-content-two h2.post-title',
           ]
        );
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-grid-wrapper-small-inner .post-grid-content-two .post-excerpt-box p',
           ]
        );		
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'meta_typography',
              'label' => esc_html__( 'Post Meta Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-grid-wrapper-small-inner .post-meta-items div',
           ]
        );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
			   'name' => 'cat_box_typography',
			   'label' => esc_html__( 'Category Box Typography', 'kiante-extra' ),
			   'selector' => '{{WRAPPER}} .post-grid-wrapper-small-inner .post-grid-content-two .category-box a',
			]
		 );


		
		$this->add_control(
          'title_length',
          [
            'label'         => esc_html__( 'Post Title Length', 'kiante-extra' ),
            'type'          => Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );

        $this->add_control(
          'content_length',
          [
            'label'         => esc_html__( 'Post Excerpt Length', 'ennlil-extra' ),
            'type'          => Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );
		

		
		$this->end_controls_section();
	
	}	
		


	protected function render() {
		
		
		$settings = $this->get_settings_for_display();

		$title_limit = $settings['title_length'];
		$content_limit = $settings['content_length'];
		
		$post_count = $settings['post_number'];
		

		
		$post_order  = $settings['post_ordering'];
		$post_sortby = $settings['post_sorting']; 
		
		$display_title = $settings['display_title'];
		$display_blog_excerpt = $settings['display_excerpt'];
		$display_blog_cat = $settings['display_cat'];
		$display_blog_author = $settings['display_author'];
		$display_blog_date = $settings['display_date'];
		

		
		$args = [
            'post_type' => 'post',
            'post_status' => 'publish',
			'order' => $settings['post_ordering'],
			'posts_per_page' => $settings['post_number'],
			'ignore_sticky_posts' => 1,
        ];
		
		// Category
        if ( ! empty( $settings['post_categories'] ) ) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }
		
		// Post Sorting
        if ( ! empty( $settings['post_sorting'] ) ) {
            $args['orderby'] = $settings['post_sorting'];
        }	
		
		// Post Offset		
		if($settings['enable_offset_post'] == 'yes') {
			$args['offset'] = $settings['post_offset_count'];
		}

		// Query
        $query = new \WP_Query( $args ); ?>

		<script type="text/javascript">
	
			jQuery(document).ready(function ($) {
				
				function rtl_slick(){
					if ($('body').hasClass("rtl")) {
					   return true;
					} else {
					   return false;
				}}
				

				/* Main Slider */
				$('.postlist-small-wrapper-carousel').slick({
					
					//infinite: true,
					fade: true,
					dots: false,
					arrows: true,
					autoplay: false,
					//focusOnSelect: true,
					
					//swipe: true,
					//swipeToSlide: true,
				
					speed: 800,
					autoplaySpeed: 2000,
				
				
					slidesToShow: 1,
					//slidesToScroll: 4,
					//rtl: rtl_slick(),

					prevArrow: '<div class="small-arrow-left"><i class="icofont-long-arrow-left"></i></div>',
					nextArrow: '<div class="small-arrow-right"><i class="icofont-long-arrow-right"></i></div>',
					
					
					responsive: [
					{
					  breakpoint: 1400,
					  settings: {
						slidesToShow: 1,
					  }
					},
					{
					  breakpoint: 1200,
					  settings: {
						slidesToShow: 1,
					  }
					},
					{
					  breakpoint: 768,
					  settings: {
						slidesToShow: 1,
					  }
					},
					{
					  breakpoint: 600,
					  settings: {
						slidesToShow: 1,
					  }
					}
				  ]
					
					

				 });
				 
				 

			});
		</script>
		
		
		<?php if ( $query->have_posts() ) : ?>
		
		
		<div class="post-list-small-wrapper">
		
			<h3>Weekly Post</h3>
			
		<div class="postlist-small-wrapper-carousel">
		
        <?php while ($query->have_posts()) : $query->the_post(); ?>
		
				<div class="post-grid-wrapper-one-inner post-grid-wrapper-small-inner">
				
					<div class="smallgrid-thumbnail-one-wrap">
						<a href="<?php the_permalink(); ?>" class="post-grid-thumbnail-small">
							<img src="<?php echo esc_attr(esc_url(get_the_post_thumbnail_url(null, 'full'))); ?>" alt="<?php the_title_attribute(); ?>">
						</a>
					</div>
				
					<div class="post-grid-content-one post-grid-content-two">
						
						<?php if($display_blog_cat=='yes'): ?>	
							<div class="category-box">
							<?php require KIANTE_THEME_DIR . '/template-parts/cat-color.php'; ?>
							</div>
						<?php endif; ?>	
						
						<?php if($display_title=='yes'): ?>
						<h2 class="post-title">
							<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), $title_limit,'') ); ?></a>
						</h2>
						<?php endif; ?>
						
						<?php if($display_blog_excerpt =='yes'): ?> 
						<div class="post-excerpt-box">
							<p><?php echo esc_html( wp_trim_words(get_the_excerpt(), $content_limit ,'') );?></p>
						</div>
						<?php endif; ?>		


						<div class="post-meta-items">
						
						<?php if($display_blog_author=='yes'): ?>

						<div class="author-name">
							<?php printf('<div class="post_grid_author_img">%1$s<a href="%2$s">%3$s</a></div>',
							get_avatar( get_the_author_meta( 'ID' ), 21 ), 
							esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ), 
							get_the_author()
							); ?>
						</div>
						<?php endif; ?>	
						
						<?php if($display_blog_date=='yes'): ?>
							<div class="date-box">
							<?php echo esc_html(get_the_date()); ?>
							</div>
						<?php endif; ?>	
						
						
						</div>

					</div>
					

				</div>
			


		<?php endwhile; ?>
		
		</div>
		
        </div>
		
		<?php wp_reset_postdata(); ?>
		
		<?php endif; ?>
		
	
		<?php 
      }
		
   
   	public function posts_cat_list() {
		
		$terms = get_terms( array(
			'taxonomy'    => 'category',
			'hide_empty'  => true
		) );

		$cat_list = [];
		foreach($terms as $post) {
		$cat_list[$post->slug]  = [$post->name];
		}
		return $cat_list;
	  
	}		
	
}


Plugin::instance()->widgets_manager->register_widget_type( new kiante_post_grid_slides() );