<?php

/**
 * Elementor Widget
 * @package Kiante
 * @since 1.0.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class theme_post_slider_main extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'posts-slider-main';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Posts Slider Main', 'kiante-extra' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'fas fa-th-large'; 
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'kiante_widgets' ];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	 
	
	protected function register_controls() {
		$this->layout_options();		
		$this->post_query_options();	
		$this->meta_options();	
		$this->design_options();
	}
	
	/**
    * Layout Options
    */
    private function layout_options() {
	
	
		$this->start_controls_section(
            'layout_option',
            [
                'label' => __( 'Layout Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		$this->add_control(
            'grid_post_column',
            [
                'label'     =>esc_html__( 'Select Grid Column', 'ennlil' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'col-lg-4',
                'options'   => [
                        'col-lg-6'    =>esc_html__( '2 Columns', 'ennlil' ),
                        'col-lg-4'    =>esc_html__( '3 Columns', 'ennlil' ),
                        'col-lg-3'    =>esc_html__( '4 Columns', 'ennlil' ),
                    ],
            ]
        );
		
		
		$this->add_responsive_control(
			'grid_img_width',
			[
				'label' =>esc_html__( 'Set Post Image Width', 'kiante-extra' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 165,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 165,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 165,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 165,
				],
				
				'selectors' => [
					'{{WRAPPER}} .post-gridstyle-two .post-grid-wrapper-two-inner .grid-thumbnail-two-wrap' => 'min-width: {{SIZE}}{{UNIT}}!important;',
				],
			]
		);		
		
		
		$this->add_responsive_control(
			'grid_img_height',
			[
				'label' =>esc_html__( 'Set Post Image Height', 'kiante-extra' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => [
					'size' => 110,
					'unit' => 'px',
				],
				'tablet_default' => [
					'size' => 110,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 110,
					'unit' => 'px',
				],
				
				'default'  => [
					'unit' => 'px',
					'size' => 110,
				],
				
				'selectors' => [
					'{{WRAPPER}} .post-gridstyle-two .post-grid-wrapper-two-inner .grid-thumbnail-two-wrap img' => 'height: {{SIZE}}{{UNIT}}!important;',
				],
			]
		);		
		
		
		$this->end_controls_section();
	
	}
	
	/**
    * Post Query Options
    */
    private function post_query_options() {
	
	
		$this->start_controls_section(
            'post_query_option',
            [
                'label' => __( 'Post Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		// Post Sort
		
        $this->add_control(
            'post_sorting',
            [
                'type'    => Controls_Manager::SELECT2,
                'label' => esc_html__('Post Sorting', 'kiante-extra'),
                'default' => 'date',
                'options' => [
					'date' => esc_html__('Recent Post', 'kiante-extra'),
                    'rand' => esc_html__('Random Post', 'kiante-extra'),
					'title'         => __( 'Title Sorting Post', 'kiante-extra' ),
                    'modified' => esc_html__('Last Modified Post', 'kiante-extra'),
                    'comment_count' => esc_html__('Most Commented Post', 'kiante-extra'),
					
                ],
            ]
        );		
		
		// Post Order
		
        $this->add_control(
            'post_ordering',
            [
                'type'    => Controls_Manager::SELECT2,
                'label' => esc_html__('Post Ordering', 'kiante-extra'),
                'default' => 'DESC',
                'options' => [
					'DESC' => esc_html__('Desecending', 'kiante-extra'),
                    'ASC' => esc_html__('Ascending', 'kiante-extra'),
                ],
            ]
        );
		
		
		// Post Categories
		
		$this->add_control(
            'post_categories',
            [
                'type'      => Controls_Manager::SELECT2,
				'label' =>esc_html__('Select Categories', 'kiante-extra'),
                'options'   => $this->posts_cat_list(),
                'label_block' => true,
                'multiple'  => true,
            ]
        );
	
		
		
		// Post Items.
		
        $this->add_control(
            'post_number',
			[
				'label'         => esc_html__( 'Number Of Posts', 'kiante-extra' ),
				'type'          => Controls_Manager::NUMBER,
				'default'       => '1',
			]
        );
		
		$this->add_control(
            'enable_offset_post',
            [
               'label' => esc_html__('Enable Skip Post', 'kiante-extra'),
               'type' => Controls_Manager::SWITCHER,
               'label_on' => esc_html__('Yes', 'kiante-extra'),
               'label_off' => esc_html__('No', 'kiante-extra'),
               'default' => 'no',
               
            ]
        );
      
        $this->add_control(
			'post_offset_count',
			  [
			   'label'         => esc_html__( 'Skip Post Count', 'kiante-extra' ),
			   'type'          => Controls_Manager::NUMBER,
			   'default'       => '1',
			   'condition' => [ 'enable_offset_post' => 'yes' ]

			  ]
        );
		
		
		$this->end_controls_section();
	
	}	
	
	/**
    * Meta Options
    */
    private function meta_options() {
	
	
		$this->start_controls_section(
            'meta_option',
            [
                'label' => __( 'Meta Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		$this->add_control(
            'display_excerpt',
            [
                'label' => esc_html__('Display Post Excerpt', 'ennlil-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'ennlil-extra'),
                'label_off' => esc_html__('No', 'ennlil-extra'),
                'default' => 'yes',
            ]
        );

		$this->add_control(
         	'display_author',
         	[
				 'label' => esc_html__('Display Author', 'kiante-extra'),
				 'type' => Controls_Manager::SWITCHER,
				 'label_on' => esc_html__('Yes', 'kiante-extra'),
				 'label_off' => esc_html__('No', 'kiante-extra'),
				 'default' => 'yes',
         	]
     	);

     	$this->add_control(
            'display_cat',
            [
                'label' => esc_html__('Display Category Name', 'kiante-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'kiante-extra'),
                'label_off' => esc_html__('No', 'kiante-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_date',
            [
                'label' => esc_html__('Display Date', 'kiante-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'kiante-extra'),
                'label_off' => esc_html__('No', 'kiante-extra'),
                'default' => 'yes',
            ]
        );
		
		$this->add_control(
            'display_read_time',
            [
                'label' => esc_html__('Display Post Read Time', 'kiante-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'kiante-extra'),
                'label_off' => esc_html__('No', 'kiante-extra'),
                'default' => 'no',
            ]
        );

	
		$this->end_controls_section();
	
	}	
	
	/**
    * Design Options
    */
    private function design_options() {
	
	
		$this->start_controls_section(
            'design_option',
            [
                'label' => __( 'Slider Typography', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'title_typography',
              'label' => esc_html__( 'Post Title Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-gridstyle-two .post-grid-content-two h3.post-title',
           ]
        );
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-gridstyle-two .post-grid-content-two .post-excerpt-box p',
           ]
        );		
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'meta_typography',
              'label' => esc_html__( 'Post Meta Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-gridstyle-two .post-grid-content-two .post-meta-items div',
           ]
        );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
			   'name' => 'author_box_typography',
			   'label' => esc_html__( 'Author Box Typography', 'kiante-extra' ),
			   'selector' => '{{WRAPPER}} .post-gridstyle-two .post-grid-content-two .post_grid_author_img',
			]
		 );


		
		$this->add_control(
          'title_length',
          [
            'label'         => esc_html__( 'Post Title Length', 'kiante-extra' ),
            'type'          => Controls_Manager::NUMBER,
            'default'       => '6',
          ]
        );

        $this->add_control(
          'content_length',
          [
            'label'         => esc_html__( 'Post Excerpt Length', 'ennlil-extra' ),
            'type'          => Controls_Manager::NUMBER,
            'default'       => '15',
          ]
        );
		

		
		$this->end_controls_section();
	
	}	
		


	protected function render() {
		
		
		$settings = $this->get_settings_for_display();

		$title_limit = $settings['title_length'];
		$content_limit = $settings['content_length'];
		
		$post_count = $settings['post_number'];
		
		$grid_post_column = $settings['grid_post_column'];
		
		
		$post_order  = $settings['post_ordering'];
		$post_sortby = $settings['post_sorting']; 
		$display_blog_excerpt = $settings['display_excerpt'];
		$display_blog_cat = $settings['display_cat'];
		$display_blog_author = $settings['display_author'];
		$display_blog_date = $settings['display_date'];
		$display_read_time = $settings['display_read_time'];

		
		$args = [
            'post_type' => 'post',
            'post_status' => 'publish',
			'order' => $settings['post_ordering'],
			'posts_per_page' => $settings['post_number'],
			'ignore_sticky_posts' => 1,
        ];
		
		// Category
        if ( ! empty( $settings['post_categories'] ) ) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }
		
		// Post Sorting
        if ( ! empty( $settings['post_sorting'] ) ) {
            $args['orderby'] = $settings['post_sorting'];
        }	
		
		// Post Offset		
		if($settings['enable_offset_post'] == 'yes') {
			$args['offset'] = $settings['post_offset_count'];
		}

		// Query
        $query = new \WP_Query( $args ); ?>
		
		<script type="text/javascript">
	
			jQuery(document).ready(function ($) {
				
				function rtl_slick(){
					if ($('body').hasClass("rtl")) {
					   return true;
					} else {
					   return false;
				}}
				

				/* Main Slider */
				$('.theme-slider-main-wrap').slick({
					
					infinite: true,
					fade: false,
					dots: false,
					arrows: true,
					autoplay: false,
					focusOnSelect: true,
					
					swipe: true,
					swipeToSlide: true,
				
					speed: 800,
					
					centerMode: true,
					centerPadding: '25%',
				
					slidesToShow: 1,
					//rtl: rtl_slick(),

					prevArrow: '<div class="slide-arrow-left"><i class="icofont-thin-left"></i></div>',
					nextArrow: '<div class="slide-arrow-right"><i class="icofont-thin-right"></i></div>',
					
					
					responsive: [
					{
					  breakpoint: 1400,
					  settings: {
						centerPadding: '15%',
					  }
					},
					{
					  breakpoint: 1200,
					  settings: {
						centerPadding: '10%',
					  }
					},
					{
					  breakpoint: 768,
					  settings: {
						centerPadding: '5%',
					  }
					},
					{
					  breakpoint: 600,
					  settings: {
						centerPadding: '0%',
					  }
					}
				  ]
					
					

				 });
				 
				 
				 $('.slick-class').on('afterChange', function(event, slick, currentSlide){
					if(currentSlide == count){
					$('.slick-class').slick('slickPause');
					setTimeout(function(){
					$('.slick-class').slick('slickGoTo',0);
					$('.slick-class').slick('slickPlay');
					},autoplaySpeed);
					}
				});
				 

			});
		</script>
		

		<?php if ( $query->have_posts() ) : ?>
		
		
		<div class="theme_post_grid__Slider theme-main-slider theme-slider-main-wrap">

		
        <?php while ($query->have_posts()) : $query->the_post(); ?>
		
		
			<div class="blog-post-slide-item">
			
				<div class="blog-post-slide-wrap" style="background-image: url(<?php echo esc_attr(esc_url(get_the_post_thumbnail_url(null, 'full'))); ?>)">
				
				</div>

				<div class="blog-post-slide-content">
					<div class="blog-post-slide-content-inner">
					<div class="category-box">
						<?php require KIANTE_THEME_DIR . '/template-parts/cat-color.php'; ?>
					</div>
								
					<h2 class="post-title">
						<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), $title_limit,'') ); ?></a>
					</h2>

					<div class="post-meta-items">
					
						<div class="author-name">
							<?php printf('%1$s<a href="%2$s">%3$s</a>',
							get_avatar( get_the_author_meta( 'ID' ), 21 ), 
							esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ), 
							get_the_author()
							); ?>
						</div>
						
						<div class="date-box">
						<?php echo esc_html(get_the_date()); ?>
						</div>
						
						<div class="comment-box">
						<?php echo get_comments_number(get_the_ID()); ?> 
						</div>
						
					</div>
					</div>
					
				</div>
				
				
        	</div>
			


		<?php endwhile; ?>
		

        </div>
		
		<?php wp_reset_postdata(); ?>
		
		<?php endif; ?>
		
	
		<?php 
      }
		
   
   	public function posts_cat_list() {
		
		$terms = get_terms( array(
			'taxonomy'    => 'category',
			'hide_empty'  => true
		) );

		$cat_list = [];
		foreach($terms as $post) {
		$cat_list[$post->slug]  = [$post->name];
		}
		return $cat_list;
	  
	}		
	
}


Plugin::instance()->widgets_manager->register_widget_type( new theme_post_slider_main() );