<?php

/**
 * Elementor Widget
 * @package Kiante
 * @since 1.0.0
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

class kiante_breaking_posts extends Widget_Base {


	/**
	 * Get widget name.
	 *
	 * Retrieve Elementor widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'posts-breaking';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Elementor widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Breaking Posts', 'kiante-extra' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Elementor widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'fas fa-th-large'; 
	}





	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Elementor widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'kiante_widgets' ];
	}

	/**
	 * Register Elementor widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	 
	
	protected function register_controls() {		
		$this->post_query_options();	
		$this->meta_options();	
		$this->design_options();
	}
	
	
	/**
    * Post Query Options
    */
    private function post_query_options() {
	
	
		$this->start_controls_section(
            'post_query_option',
            [
                'label' => __( 'Post Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'post_title',
			[
				'label'         => esc_html__( 'Breaking title', 'kiante-extra' ),
				'type'          => Controls_Manager::TEXT,
				'default'       => 'Breaking News',
			]
        );
		
		
		// Post Sort
		
        $this->add_control(
            'post_sorting',
            [
                'type'    => Controls_Manager::SELECT2,
                'label' => esc_html__('Post Sorting', 'kiante-extra'),
                'default' => 'date',
                'options' => [
					'date' => esc_html__('Recent Post', 'kiante-extra'),
                    'rand' => esc_html__('Random Post', 'kiante-extra'),
					'title'         => __( 'Title Sorting Post', 'kiante-extra' ),
                    'modified' => esc_html__('Last Modified Post', 'kiante-extra'),
                    'comment_count' => esc_html__('Most Commented Post', 'kiante-extra'),
					
                ],
            ]
        );		
		
		// Post Order
		
        $this->add_control(
            'post_ordering',
            [
                'type'    => Controls_Manager::SELECT2,
                'label' => esc_html__('Post Ordering', 'kiante-extra'),
                'default' => 'DESC',
                'options' => [
					'DESC' => esc_html__('Desecending', 'kiante-extra'),
                    'ASC' => esc_html__('Ascending', 'kiante-extra'),
                ],
            ]
        );
		
		
		// Post Categories
		
		$this->add_control(
            'post_categories',
            [
                'type'      => Controls_Manager::SELECT2,
				'label' =>esc_html__('Select Categories', 'kiante-extra'),
                'options'   => $this->posts_cat_list(),
                'label_block' => true,
                'multiple'  => true,
            ]
        );
	
		
		
		// Post Items.
		
        $this->add_control(
            'post_number',
			[
				'label'         => esc_html__( 'Number Of Posts', 'kiante-extra' ),
				'type'          => Controls_Manager::NUMBER,
				'default'       => '1',
			]
        );
		
		$this->add_control(
            'enable_offset_post',
            [
               'label' => esc_html__('Enable Skip Post', 'kiante-extra'),
               'type' => Controls_Manager::SWITCHER,
               'label_on' => esc_html__('Yes', 'kiante-extra'),
               'label_off' => esc_html__('No', 'kiante-extra'),
               'default' => 'no',
               
            ]
        );
      
        $this->add_control(
			'post_offset_count',
			  [
			   'label'         => esc_html__( 'Skip Post Count', 'kiante-extra' ),
			   'type'          => Controls_Manager::NUMBER,
			   'default'       => '1',
			   'condition' => [ 'enable_offset_post' => 'yes' ]

			  ]
        );
		$this->end_controls_section();
		
		$this->start_controls_section(
            'styling_section',
            [
                'label' => esc_html__('Styling Settings', 'kiante-extra'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control('bg_color', [
            'label' => esc_html__('Bg color', 'kiante-extra'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .breaking-ttile" => "background: {{VALUE}}",
            ]
        ]);
		$this->add_control('main_text_color', [
            'label' => esc_html__('Main Text color', 'kiante-extra'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .breaking-news-post-title a" => "color: {{VALUE}}",
            ]
        ]);
		$this->add_control('border_color', [
            'label' => esc_html__('Border color', 'kiante-extra'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
				"{{WRAPPER}} .breaking-posts-wrapper" => "border: 1px solid {{VALUE}}",
            ]
        ]);
		$this->end_controls_section();
	
	}	
	
	/**
    * Meta Options
    */
    private function meta_options() {
	
	
		$this->start_controls_section(
            'meta_option',
            [
                'label' => __( 'Meta Options', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		

		
		$this->add_control(
            'display_date',
            [
                'label' => esc_html__('Display Date', 'kiante-extra'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'kiante-extra'),
                'label_off' => esc_html__('No', 'kiante-extra'),
                'default' => 'yes',
            ]
        );
		

	
		$this->end_controls_section();
	
	}	
	
	/**
    * Design Options
    */
    private function design_options() {
	
	
		$this->start_controls_section(
            'design_option',
            [
                'label' => __( 'Slider Typography', 'kiante-extra' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
		
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'title_typography',
              'label' => esc_html__( 'Post Title Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-list-content-one h2.post-title',
           ]
        );
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'excerpt_typography',
              'label' => esc_html__( 'Post Excerpt Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-list-content-one .post-excerpt-box p',
           ]
        );		
		
		$this->add_group_control(
           Group_Control_Typography::get_type(),
           [
              'name' => 'meta_typography',
              'label' => esc_html__( 'Post Meta Typography', 'kiante-extra' ),
              'selector' => '{{WRAPPER}} .post-list-content-one .post-meta-items div',
           ]
        );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
			   'name' => 'cat_box_typography',
			   'label' => esc_html__( 'Category Box Typography', 'kiante-extra' ),
			   'selector' => '{{WRAPPER}} .post-list-content-one .post-meta-items .category-box a',
			]
		 );


		
		$this->add_control(
          'title_length',
          [
            'label'         => esc_html__( 'Post Title Length', 'kiante-extra' ),
            'type'          => Controls_Manager::NUMBER,
            'default'       => '20',
          ]
        );

		

		
		$this->end_controls_section();
	
	}	
		


	protected function render() {
		
		
		$settings = $this->get_settings_for_display();

		$title_limit = $settings['title_length'];
		
		$post_count = $settings['post_number'];
		
		
		
		$post_order  = $settings['post_ordering'];
		$post_sortby = $settings['post_sorting']; 
		

		$display_blog_date = $settings['display_date'];
		

		
		$args = [
            'post_type' => 'post',
            'post_status' => 'publish',
			'order' => $settings['post_ordering'],
			'posts_per_page' => $settings['post_number'],
			'ignore_sticky_posts' => 1,
        ];
		
		// Category
        if ( ! empty( $settings['post_categories'] ) ) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }
		
		// Post Sorting
        if ( ! empty( $settings['post_sorting'] ) ) {
            $args['orderby'] = $settings['post_sorting'];
        }	
		
		// Post Offset		
		if($settings['enable_offset_post'] == 'yes') {
			$args['offset'] = $settings['post_offset_count'];
		}

		// Query
        $query = new \WP_Query( $args ); ?>


		<script type="text/javascript">
	
			jQuery(document).ready(function ($) {
				
				function rtl_slick(){
					if ($('body').hasClass("rtl")) {
					   return true;
					} else {
					   return false;
				}}
				

				/* Main Slider */
				$('.breaking-news-carosuel').slick({
					
					//infinite: true,
					fade: true,
					dots: false,
					arrows: false,
					autoplay: false,
					//focusOnSelect: true,
					
					//swipe: true,
					//swipeToSlide: true,
				
					speed: 800,
					autoplaySpeed: 2000,
				
				
					slidesToShow: 1,
					//slidesToScroll: 4,
					//rtl: rtl_slick(),

					prevArrow: '<div class="grid-arrow-left"><i class="icofont-long-arrow-left"></i></div>',
					nextArrow: '<div class="grid-arrow-right"><i class="icofont-long-arrow-right"></i></div>',
					
					
					responsive: [
					{
					  breakpoint: 1400,
					  settings: {
						slidesToShow: 1,
					  }
					},
					{
					  breakpoint: 1200,
					  settings: {
						slidesToShow: 1,
					  }
					},
					{
					  breakpoint: 768,
					  settings: {
						slidesToShow: 1,
					  }
					},
					{
					  breakpoint: 600,
					  settings: {
						slidesToShow: 1,
					  }
					}
				  ]
					
					

				 });
				 
				 

			});
		</script>






		<?php if ( $query->have_posts() ) : ?>
		
		
		<div class="breaking-posts-wrapper">
		
		<div class="breaking-ttile">
			<?php echo esc_html($settings["post_title"]); ?>
		</div>
		
		<div class="breaking-news-carosuel">
		
        <?php while ($query->have_posts()) : $query->the_post(); ?>
		
				
				<div class="breaking-news-inner">
				
					<div class="breaking-news-post-title">
						<a href="<?php the_permalink(); ?>"><?php echo esc_html( wp_trim_words(get_the_title(), $title_limit,'') ); ?></a>
					</div>
					
					<?php if($display_blog_date=='yes'): ?>
					<div class="breaking-news-post-date">
					<?php echo esc_html(get_the_date( 'F j' )); ?>
					</div>
					<?php endif; ?>
					
				</div>
			
		<?php endwhile; ?>
		
        </div>
		
		</div>
		
		<?php wp_reset_postdata(); ?>
		
		<?php endif; ?>
		
	
		<?php 
      }
		
   
   	public function posts_cat_list() {
		
		$terms = get_terms( array(
			'taxonomy'    => 'category',
			'hide_empty'  => true
		) );

		$cat_list = [];
		foreach($terms as $post) {
		$cat_list[$post->slug]  = [$post->name];
		}
		return $cat_list;
	  
	}		
	
}


Plugin::instance()->widgets_manager->register_widget_type( new kiante_breaking_posts() );